/**
 * @file bacore.hpp
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief Core library info and lifecycle API
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include "bacore.h"
#include "version.hpp"

namespace brainaccess::core
{
    /**
     * @brief Error codes for init
     */
    enum class init_error: ba_init_error
    {
        UNKNOWN = BA_INIT_ERROR_UNKNOWN,                          ///< Unknown error while calling `core::init()`
        OK = BA_INIT_ERROR_OK,                                    ///< Initialized successfully
        CONFIG_TYPE = BA_INIT_ERROR_CONFIG_TYPE,                  ///< Configuration file contains a value of the wrong type
        CONFIG_PARSE = BA_INIT_ERROR_CONFIG_PARSE,                ///< Configuration file could not be parsed, invalid JSON or missing value
        INCOMPATIBLE_VERSION = BA_INIT_ERROR_INCOMPATIBLE_VERSION ///< The version requested and the version 
    };

    /**
     * @brief Initializes the library. Must be called before any other BrainAccess
     * Core library function. Only call once.
     * 
     * @details This function reads the config file, starts logging, etc. It first
     * checks if the version of the library that the application expects and the
     * version of the library installed are compatible.
     * 
     * @param version The version of the library that the application expects.
     * @return Error code
     */
    [[maybe_unused]] static init_error init(const version& version) noexcept
    {
        return static_cast<init_error>(ba_core_init(&version));
    }
    
    /**
     * @brief Returns the installed library's actual version
     * @details Uses semantic versioning
     * 
     * @return Installed library's actual version
     */
    [[maybe_unused]] [[nodiscard]] static const version& get_version() noexcept
    {
        return *ba_core_get_version();
    }
    
    /**
     * @brief Closes the library and cleans up afterwards. Must be called after all
     * BrainAccess Core libary functions used by the application. Only call once.
     * 
     * @details If `core::init()` failed, do not call this function.
     */
    [[maybe_unused]] static void close() noexcept
    {
        return ba_core_close();
    }
}
